/*!
  * \file				: sysdata.h
  * \brief				: Header for sysdata.c file.
  *						  This file contains the common defines of the application.
  *
  */

#ifndef __SYS_DATA_H
#define __SYS_DATA_H

#ifdef __cplusplus
extern "C" {
#endif

#include "stdint.h"

#define CONCAT(a, b) CONCAT_INNER(a, b)											// These three macros
#define CONCAT_INNER(a, b) a ## b												// generate unique variables
#define UNIQUE_NAME(base) CONCAT(base, __COUNTER__)								// according to template "baseX", like "base1", "base2" and etc.

#define SAMPLE_ARRAY_SIZE 32

//------------------------------------------------------------------------------

typedef union
{
	uint16_t w;
	int16_t  sw;
	uint8_t  b[2];
	int8_t   sb[2];
} bword_t;

//------------------------------------------------------------------------------
// This structure must be aligned to 4-byte address
typedef struct
{
	// Device spezifisch / Einstellungsbereich
	uint16_t device_type;														// 0	ECS interne Gertetyp Kennung (IHP 500 -> daher wird angedacht die Gerätekennung 500 zu nehmen)
	uint16_t UNIQUE_NAME(reserved);												// 1	Reserviert wg Alignement
	uint32_t sn;																// 2-3	Seriennummer Teil 1 -> wird bei der Produktion geschrieben

	uint16_t fw_revision_major;													// 4	Firmware Revision Major -> wird bei einer neuen Gertegeneration hochgezhlt
	uint16_t fw_revision_minor;													// 5	Firmware Revision Minor -> wird hochegezhlt, wenn nderungen in der Firmware auch nderungen in der Bedienungsanleitung zur Folge haben

	uint16_t fw_revision_revision;												// 6	Firmware Revision Revision -> wird bei jeder nderung der Firmware hochgezhlt
	uint16_t UNIQUE_NAME(reserved);												// 7	Reserviert wg Alignement

	uint32_t baudrate;															// 8-9	Die Modbus Baudrate (default 19200 Bit/s). Gltig im Bereich von 50 Bit/s bis 1000 000 Bit/s

	uint16_t parityMode;														// 10	UART Parity Mode (default EVEN) mgliche Werte 0=EVEN, 1=ODD, 2=NONE
	uint16_t stopBit;															// 11	UART Stop Bit Reserviert, Aktuell automatisch selektiert nach Modbus standart Bei Parity EVEN oder ODD 1, bei NONE werden 2 stopbits genutzt

	uint16_t slave_address;														// 12	Modbus Slave Adresse (default 1) mgliche Werte von 1 bis 247
	uint16_t command;															// 13	Kommando Word, das Schreiben bestimmter Werte in dieses Register fhrt bestimmte Aktionen aus

	uint16_t ibn_day;															// 14	Inbetriebnahme Datum: Tag
	uint16_t ibn_month;															// 15	Inbetriebnahme Datum: Monat
	uint16_t ibn_year;															// 16	Inbetriebnahme Datum: Jahr
	uint16_t user_id;															// 17	Freies Feld für eine Kunden ID oder ähnlich
	uint16_t test[8];											// 18-25	RESERVIERT

	//--- Konfiguration Schaltausgänge ---
	uint16_t lvpStart;															// 26	Spannung ab der die LOW Voltage Protection aktiv wird in [mV]
	uint16_t lvpStop;															// 27	Spannung ab der die LOW Voltage Protection wieder inaktiv wird [mV]

	uint16_t ovpStart;															// 28	Spannung ab der die OVER Voltage Protection aktiv wird in [mV]
	uint16_t ovpStop;															// 29	Spannung ab der die OVER Voltage Protection wieder inaktiv wird [mV]

	int16_t loadCurrentLimit;													// 30	maximaler Laststrom in A wenn der Strom grer ist als der eingestelle Wert dann wird die Laststrom Protection aktiv, darf nicht unsigned sein, da Entladestrom mit Minus angegeben [A] Default -500 A
	int16_t chargeCurrentLimit;													// 31	maximaler Ladestrom in A wenn der Strom grer ist als der eingestelle Wert dann wird die Ladestrom Protection aktiv [A] Default:500A


	int16_t chargeStopHighTemperatureStart;										// 32	Abschalttemperatur Ladung wegen zu hoher Temperatur [C * 100]
	int16_t chargeStopHighTemperatureStop;										// 33	Wiedereinschalttemperatur [C * 100]

	int16_t chargeStopLowTemperatureStart;										// 34	Abschalttemperatur Ladung wegen zu niedriger Temperatur [C * 100]
	int16_t chargeStopLowTemperatureStop;										// 35	Wiedereinschalttemperatur [C * 100]

	int16_t dischargeStopHighTemperatureStart;									// 36	Abschalttemperatur Entladung wegen zu hoher Temperatur [C * 100]
	int16_t dischargeStopHighTemperatureStop;									// 37	Wiedereinschalttemperatur[C * 100]

	int16_t dischargeStopLowTemperatureStart;									// 38	Abschalttemperatur EntLadung wegen zu niedriger Temperatur
	int16_t dischargeStopLowTemperatureStop;									// 39	Wiedereinschalttemperatur


	//ah-Counter
	uint16_t peukert;															// 40		Peukert Faktor (Wert für den Exponent) * 100
	uint16_t cef;																// 41		Ladeeffektivitätsfaktor in %

	uint16_t iBatFull;															// 42		Strom in % ab dem die Zellen als voll erkannt werden (Strom muss für die Zeit timeBatFull unter den Wert fallen)
	uint16_t tBatFull;															// 43		Zeit in Sekunden ab dem die Zellen als voll erkannt werden (Strom muss unter dem Wert iBattFull liegen)

	uint16_t uBatFull;															// 44		Spannung für Erkennung Battery voll in [mV]
	uint16_t uBatEmpty;															// 45		Reserviert
	uint32_t cellCapacity;														// 46-47	Die Zellkapazität in mAh (bei parallel geschalteten Zellen wird die Kapazitt der Zellen addiert)
	uint32_t battEnergy;														// 48-49	Die Zellkapazität in mWh (Wh = mAh * Vnom)
	uint16_t socCalcMode;														// 50		SoC calculation method: 0 (simple - default) - based on Ah values; 1 (advanced) - based on Wh values
	uint16_t UNIQUE_NAME(reserved);												// 51		RESERVIERT
	uint16_t cellRatedDischargeTime;											// 52		Standard discharge time in h
	int16_t  uBatEmptyCompStartTemp;											// 53		We start calculating uBatEmpty compensations only when cell temperature is lower than this value
	int16_t  uBatEmptyCompStopTemp;												// 54		We stop calculating uBatEmpty compensations when cell temperature is lower than this value
	uint16_t uBatEmptyCompStopVolt;												// 55		uBatEmpty Voltage at temperatures lower than lvpCompStopTemp
	int16_t	 extraDischargeStrom_mA;											// 56
	uint16_t cefW;																// 57		CeF for Wh calculations
	uint16_t lvpMode;															// 58-59	RESERVIERT
	uint16_t ovpMode;															// 58-59	RESERVIERT

	// ------- Calibration Parameter -----
	int32_t batteryCurrentOffset;												// 60-61	Offset bei Raumtemperatur ADC Schritte
	int16_t batteryCurrentOffsetRefTemperatureShunt;							// 62		gesicherte Shunt Temperatur zum Zeitpunkt der Kalibrierung
	int16_t batteryCurrentOffsetRefTemperatureChip;								// 63		gesicherte Chip Temperatur zum Zeitpunkt der Kalibrierung
	int32_t batteryCurrentOffsetRefshuntVoltage;								// 64-65	gesicherte Shunt Spannung zum Zeitpunkt der Kalibrierung

	int32_t batteryCurrentOffsetCommonModeCalibrationVoltage;					// 66-67	gesicherte Spannuung am Shunt für Common Mode Error Compensation
	int32_t batteryCurrentOffsetCommonModeCompensationFactor;					// 68-69	Common Mode Error Compensation Faktor Faktor pro mV Shuntspannungsnderung * 1000

	int16_t batteryCurrentOffsetTemperatureCalibrationTemperature;				// 70		gesicherte Temperatur zum Zeitput der Offset Temperatur Kompensation
	int16_t UNIQUE_NAME(reserved);												// 71		reserviert
	int32_t batteryCurrentOffsetTemperatureCompensationFactor;					// 72-73	Berechnete Temperatur Kompensation für Strom Offset

	int32_t batteryCurrentGainCorrectionFaktor;									// 74-75	Gain Korrektur Faktor * 1000000
	int16_t batteryCurrentGainRefTempShunt;										// 76		Gesicherte Temperatur Shunts bei Gain Kallibrierung
	int16_t batteryCurrentGainRefTempChip;										// 77		Gesicherte Temperatur Chip bei Gain Kallibrierung
	int32_t batteryCurrentGainRefCurrent;										// 78-79	Referenz Strom für die Gain Kalibrierung, muss vor dem Start des Kal. Befehls hier eingegeben werden

	int16_t batteryCurrentGainTemperatureCalibrationShuntTemperature;			// 80		Gesicherte Shunt Temperatur bei Start der Gain Temperatur Kompensation
	int16_t UNIQUE_NAME(reserved);												// 81		reserviert
	int32_t batteryCurrentGainTemperatureCompensationShuntFactor;				// 82-83	Berechneter Faktor für Gain Temperatur kompensation. Pro 1/100 C * 1000000000
	int16_t batteryCurrentGainTemperatureCalibrationChipTemperature;			// 84		Gesicherte Chip Temperatur bei Start der Gain Temperatur Kompensation
	int16_t UNIQUE_NAME(reserved);												// 85		reserviert
	int32_t batteryCurrentGainTemperatureCompensationChipFactor;				// 86-87	zustzlicher Kalibrierungsfaktor, nicht in Verwendung

	uint32_t UNIQUE_NAME(reserved)[6];

} device_parameter_t;

//------------------------------------------------------------------------------
// This structure must be aligned to 4-byte address
typedef struct
{	/*offset 0*/
	/* 2 of 8 */ uint16_t voltageVref;											// 100		Aktuelle Spannung der Referenzspannungsquelle des ADS1260 gemessen von dem STM32G0 [mV]
	/* 4 of 8 */ uint16_t realVdd;												// 101		Aktuelle Umrechnung von gemessene externe RefSpannung auf ADC Ref Spannung = Vdd, Vdd ist die Referenz des internen ADC. Diese wird jetzt als Referenz für die interne ADC Messungen verwendet [mV]
	/* 6 of 8 */ int16_t  chipTemperature;										// 102		Chiptemperatur gemessen auf dem STM32G0 [C * 100]
	/* 8 of 8 */ int16_t  shuntTemperature;										// 103		Gemessene Shunt Temperatur [C * 100]
	/* 2 of 8 */ int16_t  shuntTemperatureMax;									// 104		Höchste Shunttemperatur seit Programmstart, kann durch Schreiben von Int16-Min (-32.768) in das Register zurückgesetzt werden
	/* 4 of 8 */ int16_t  shuntTemperatureMin;									// 105		Niedrigste Shunttemperatur seit Programmstart, kann durch Schreiben von  Int16_Max (+32.767) in das Register zurückgesetzt werden
	/* 8 of 8 */ uint32_t shuntVoltage;											// 106-107	Shuntspannung in mV (Common Mode Voltage) gemessen von dem STM32G0. Spannung zwischen Shunt B und GND Versorgung
	/* 4 of 8 */ int32_t  batteryVoltage;										// 108-109!	Gemessenen Batteriespannung in mV. Es kann in beide Richtungen gemessen werden (mindestens von -60V bis +60V). Es muss darauf geachtet werden, dass die gemessen Spannung positiv ist (+ an Batterieplus und - an Batterieminus)
	/* 8 of 8 */ int32_t  batteryVoltageMax;									// 110-111	Hchster Spannungswert in mV (Batteriespannung) seit Programmstart. Kann durch schreiben von int32-Min (-2,147,483,648) in das Register zurckgesetzt werden
	/* 4 of 8 */ int32_t  batteryVoltageMin;									// 112-113	Niedrigster Spannungswert in mV (Batteriespannung) seit Programmstart. Kann durch Schreiben von  int32-Max (2,147,483,647) in das Register zurckgesetzt werden
	/* 8 of 8 */ int32_t  batteryCurrent;										// 114-115!	Gemessener Batteriestrom in mA. Es kann in beide Richtungen gemessen werden, es muss darauf geachtet werden, dass der Stromsensor so eingebaut wird, dass der Ladestrom positiv gemessen wird und der Laststrom negativ, ansonsten falsche SOC Berechnung
	/* 4 of 8 */ int32_t  batteryCurrentMax;									// 116-117	Hchster Batteriestrom seit Programmstart in mA. Kann durch Schreiben von Int32-Min (-2,147,483,648) in das Register zurückgesetzt werden
	/* 8 of 8 */ int32_t  batteryCurrentMin;									// 118-119	Niedrigster Batteriestrom seit Programmstart in mA. Kkann durch Schreiben von Int32-Max (2,147,483,647) in das Register zurückgesetzt werden
	/* 4 of 8 */ int32_t  batteryCurrentCorrected;								// 120-121!	Gemessener Batteriestrom mit berücksichtitgten Peukert Faktor in mA
	/* 8 of 8 */ int32_t  mAhCounter;											// 122-123	Kapazitätszähler in [mAh] umgerechnet von mAsCounter
	/* 8 of 8 */ int64_t  mAsCounter;											// 124-127!	Kapazitätszähler in mAhs, zählt den gemessenen Strom, positiv für Ladeströme und negativ für Lastströme (Mit Peukert und CEF)
	/* 4 of 8 */ int32_t  soc;													// 128-129!	State of Charge in % * 1000
	/* 6 of 8 */ int16_t  ovpState;												// 130		Zustand Schaltausgang OVP
	/* 8 of 8 */ int16_t  lvpState;												// 131		Zustand Schaltausgang LVP
	/* 4 of 8 */ int32_t  batteryPower;											// 132-133!	Leistung Batterie[mW]
	/* 8 of 8 */ int32_t  detectedCapacity;										// 134-135	Festgestellte Kapazität [mAh]
	/* 4 of 8 */ int32_t  detectedEnergy;										// 136-137	[mWh]
	/* 6 of 8 */ int16_t  SoH;													// 138		[%]
	/* 8 of 8 */ int16_t  efficiency;											// 139		[%]
	/* 2 of 8 */ int16_t  selfDischarge;										// 140		[mA]
	/* 4 of 8 */ int16_t  selfDischargeAvg;										// 141		[mA]
	/* 6 of 8 */ int16_t  esr;													// 142		[mOhm]
	/* 8 of 8 */ int16_t  esr_fast;												// 143		[mOhm]
	/* 6 of 8 */ uint32_t lastTimeVbatFull;										// 144-145	[s]
	/* 8 of 8 */ uint32_t lastTimeVbatEmpty;									// 146-147	[s]
	/* 4 of 8 */ uint32_t onTime;												// 148-149	[s] Runtime Zähler zum speichern der Statistik Werte im EEPROM und als Betriebszeitzähler
	/* 8 of 8 */ uint32_t dischargeTotalAh;										// 150-151	[mAs]
	/* 4 of 8 */ uint32_t chargeTotalAh;										// 152-153	[mAs]
	/* 8 of 8 */ uint32_t dischargeTotalWh;										// 154-155	[mAs]
	/* 4 of 8 */ uint32_t chargeTotalWh;										// 156-157	[mAs]
	/* 6 of 8 */ uint16_t fullCyclesCnt;										// 158
	/* 8 of 8 */ uint16_t volatile UNIQUE_NAME(reserved);								// 159
	/* 8 of 8 */ int64_t  mAs_AutoMode;											// 160-163
	/* 8 of 8 */ int64_t  mWs_AutoMode;											// 164-167
	/* 2 of 8 */ uint16_t ovp_sense;                                                                         // 168
	/* 4 of 8 */ uint16_t lvp_sense;                                                                        // 169
	/* 6 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 170
	/* 8 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 171
	/* 2 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 172
	/* 4 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 173
	/* 6 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 174
	/* 8 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 175
	/* 2 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 176
	/* 4 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 177
	/* 6 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 178
	/* 8 of 8 */ uint16_t UNIQUE_NAME(reserved);								// 179

	// --- DEBUG WERTE
	/* 4 of 8 */ int32_t  mAh_AutoMode;											// 180-181	[mWh]
	/* 8 of 8 */ int32_t  mWh_AutoMode;											// 182-183	[mWh]
	/* 8 of 8 */ int64_t  mWsCounter;											// 184-187	[mWs]
	/* 4 of 8 */ int32_t  mWhCounter;											// 188-189	[mWh]
	/* 8 of 8 */ int32_t  correctedStromForChargeWhCnt;							// 190-191	Reserviert
	/* 2 of 8 */ uint16_t uBatEmptyTempComp;									// 192		[mV]
	/* 4 of 8 */ uint16_t adc_restarts;											// 193
	/* 6 of 8 */ int16_t  UNIQUE_NAME(reserved);								// 194		Reserviert
	/* 8 of 8 */ int16_t  UNIQUE_NAME(reserved);								// 195		Reserviert
	/* 4 of 8 */ int32_t  battryCurrentRaw;										// 196-197	Gemessener Strom in ADC Schritte Ohne Korrekturen
	/* 8 of 8 */ int32_t  batteryCurrentOffsetCommonModeCorrectionADCSteps;		// 198-199	Offset Korrektur Wert aufgrund Common Mode Error
	/* 4 of 8 */ int32_t  fast_current;											// 200-201
	/* 6 of 8 */ int32_t  fast_voltage;											// 202-203
	/* 8 of 8 */ int16_t  esrCalcStatus;										// 204
	/* 2 of 8 */ uint16_t esrCalcTime;											// 205
	/* 4 of 8 */ int16_t  selfDischargeStatus;									// 206
	/* 6 of 8 */ uint16_t selfDischargeTime;									// 207
	/* 8 of 8 */ uint8_t  message[24];											// 208-219	ASCII-char message, limited to 8 letters
	/* 4 of 8 */ int32_t  current_buffer[SAMPLE_ARRAY_SIZE];					// 220-283
	/* 8 of 8 */ int32_t  voltage_buffer[SAMPLE_ARRAY_SIZE];					// 284-347
	/* 4 of 8 */ int32_t  current_buffer_fast[SAMPLE_ARRAY_SIZE];				// 348-411
	/* 8 of 8 */ int32_t  voltage_buffer_fast[SAMPLE_ARRAY_SIZE];				// 412-475

} device_values_t;

//------------------------------------------------------------------------------

typedef struct
{
	device_parameter_t parameter;												// 0-99
	device_values_t    values;													// 100
} sys_data_struct_t;

//------------------------------------------------------------------------------
typedef volatile union
{
	sys_data_struct_t s;
	bword_t mb[sizeof(sys_data_struct_t) / 2];
} sys_data_t;

//------------------------------------------------------------------------------

extern sys_data_t sys_data;

//------------------------------------------------------------------------------

void SYS_DATA_Init(void);

#ifdef __cplusplus
}
#endif

#endif // __SYS_DATA_H