/**
  ******************************************************************************
  * @file    bms.c
  * @author  ECS, Zed Kazharov
  * @version V1.0.0
  * @date    05-Jan-2023
  * @brief   BMS Modul 
  *          Beschreibung in Header
  ******************************************************************************
  */ 

// --- INCLUDES ----------------------------------------------------------------
#include "bms.h"
#include "main.h"
#include "sysdata.h"
#include "output.h"
#include "led.h"
#include "stdio.h"
#include "eeprom.h"

//--- GGF. EXTERNE VARIABLEN ---------------------------------------------------  

//--- LOKALE DEFINES - bitte hier dokumentieren --------------------------------
#define LVP_TIME_INTERVALL 1000 //Einheit fr Delay ist Sekunden, deshalb Zhler alle 1000ms hochzhlen
#define MINUTE_TIME_INTERVALL 60000 
#define HOUR_TIME_INTERVALL (60UL * 60000UL) 
//--- LOKALE TYPE DEFS - bitte hier dokumentieren-------------------------------

//--- DEFINATIONEN GLOBALER VARIABLEN - Bitte in Header dokumentieren ----------

//--- LOKALE VARIABLEN - bitte hier dokumentieren ------------------------------
uint32_t lastLvpDelayTime = 0;
uint32_t lastMinuteTime = 0;
uint32_t lastHourTime = 0;
uint32_t batteryState = 0; // 0 = discharged, 1 = charged
uint32_t initRefresh = 1;
//--- LOKALE FUNKTIONS PROTOTYPEN ---------------------------------------------- 


//--- LOKALE FUNKTIONEN - bitte hier dokumentieren -----------------------------
/**
  * @brief  Calculate Setpoints
  *
  *         Beschreibung:
  *         Berechnete die aktullen Schwellwerte anhand der Temperatur
  *         
  *
  *   
  * @param  Keine
  *              
  * @retval Kein
  */
void calculateSetpoints()
{
  int32_t delta_t;
  
  delta_t = sysData.s.cellTemperature - sysData.s.refTemp;
  delta_t /= 10;
  
  sysData.s.ovpAlarmComp = sysData.s.ovpAlarm + delta_t * sysData.s.ovpAlarmTempco;
  sysData.s.lvpAlarmComp = sysData.s.lvpAlarm + delta_t * sysData.s.lvpAlarmTempco;
  sysData.s.ovpStartComp = sysData.s.ovpStart + delta_t * sysData.s.ovpStartTempco;  
  sysData.s.ovpStopComp = sysData.s.ovpStop + delta_t * sysData.s.ovpStopTempco;  
  sysData.s.lvpStartComp = sysData.s.lvpStart + delta_t * sysData.s.lvpStartTempco;  
  sysData.s.lvpStopComp = sysData.s.lvpStop + delta_t * sysData.s.lvpStopTempco;  
  sysData.s.vShutdownStartComp = sysData.s.vShutdownStart +  delta_t * sysData.s.vShutdownStartTempco;  
  sysData.s.vShutdownStopComp = sysData.s.vShutdownStop + delta_t * sysData.s.vShutdownStopTempco;    

  sysData.s.balancerVoltageComp =  sysData.s.defaultBalVoltage + delta_t * sysData.s.balancerVoltageTempco;      


  if (!sysData.s.masterMode) 
  {     
    sysData.s.balancerVoltage = sysData.s.balancerVoltageComp;  
  }

}
//--- GLOBALE FUNKTIONEN - bitte in Header dokumentieren------------------------
void BMS_Exec(void)
{
  calculateSetpoints();


  // ------------ ber Temperatur berwachung ----------------------- 
  if (sysData.s.cellTemperature > sysData.s.otShutdownStart)
  {
    sysData.s.otShutdown = 1; 
  }
  
  if (sysData.s.cellTemperature < sysData.s.otShutdownStop)
  {
    sysData.s.otShutdown = 0;
  }

    // ------------ Unter Temperatur berwachung ----------------------- 
  if (sysData.s.cellTemperature < sysData.s.utpChargeStart)
  {
    sysData.s.utChargeShutdown = 1; 
  }
  
  if (sysData.s.cellTemperature > sysData.s.utpChargeStop)
  {
    sysData.s.utChargeShutdown = 0;
  }

  if (sysData.s.cellTemperature < sysData.s.utpDischargeStart)
  {
    sysData.s.utDischargeShutdown = 1; 
  }
  
  if (sysData.s.cellTemperature > sysData.s.utpDischargeStop)
  {
    sysData.s.utDischargeShutdown = 0;
  }
  
  // ------------ VOLTAGE SHUTDOWN berwachung ----------------------- 
  if (sysData.s.cellVoltage < sysData.s.vShutdownStartComp)
  {
    sysData.s.voltageShutdown=1;
  }
  
  if (sysData.s.cellVoltage > sysData.s.vShutdownStopComp)
  {
    sysData.s.voltageShutdown=0;
  }
                  
  
  // -------------- OVP ----------------------------------------
  // Spannung zu hoch? Temperatur nicht OK? --> OVP Sicherheitsschleife fffen!
  if ((sysData.s.cellVoltage > sysData.s.ovpStartComp) || (sysData.s.otShutdown == 1 ) || (sysData.s.utChargeShutdown == 1 ) || (sysData.s.voltageShutdown  == 1) )
  {
    OUTPUT_OVP_Enable();
    sysData.s.ovpState = 1;
  }
  
  // Spannung und Temperaturen wieder OK? --> OVP Sicherheitsschleife wieder schlieen! 
  if ((sysData.s.cellVoltage  < sysData.s.ovpStopComp) &&  (sysData.s.otShutdown == 0  ) &&  (sysData.s.utChargeShutdown == 0  ) && (sysData.s.voltageShutdown  == 0) )
  {
    OUTPUT_OVP_Disable();
    sysData.s.ovpState = 0;
  }
  
  // LVP
  // Spannung zu niedrig? --> lVP Sicherheitsschleife verzgert fffen!
  if (sysData.s.cellVoltage < sysData.s.lvpStartComp) 
  {
    if (HAL_GetTick() >= lastLvpDelayTime + LVP_TIME_INTERVALL)
    {
      if (sysData.s.lvpDelayCounter < sysData.s.lvpTime) sysData.s.lvpDelayCounter++; //TODO : Prfe Zeitzhler 
      lastLvpDelayTime = HAL_GetTick();
    }
  }
  
  if ((sysData.s.lvpDelayCounter == sysData.s.lvpTime) || ( sysData.s.otShutdown == 1 ) || (sysData.s.utDischargeShutdown == 1))
  {
    OUTPUT_LVP_Enable();
    sysData.s.lvpState = 1;
  }	
  
  

  // Spannung wieder OK? --> LVP Sicherheitsschleife wieder schlieen! 
  if (sysData.s.cellVoltage > sysData.s.lvpStopComp)  
  {
    sysData.s.lvpDelayCounter=0;
  }
  
  if ((sysData.s.lvpDelayCounter == 0) &&  (sysData.s.otShutdown == 0 ) && (sysData.s.utDischargeShutdown == 0))
  {
    OUTPUT_LVP_Disable();
    sysData.s.lvpState = 0;
  }	
  
  // ------------ ALARM LED -------------------------
  static uint32_t ledErrorBlinkState=0;
  if (sysData.s.converterError == CONVERTER_ERROR_OUTPUT_VOLT_TOO_HIGH)
  {
    if  (ledErrorBlinkState != 1)
    {
      LED_ErrorStartBlink(7000, 1000); // 7 sekunden
      ledErrorBlinkState = 1;
    }
    
  }
  else if ((sysData.s.converterError == CONVERTER_ERROR_OUTPUT_VOLT_TOO_LOW) || (sysData.s.converterError == CONVERTER_ERROR_STARTUP_ERROR))
  {
    if  (ledErrorBlinkState != 2)
    {
      LED_ErrorStartBlink(5000, 1000); // 5 sekunden
      ledErrorBlinkState = 2;
    }
  }
  else if (sysData.s.otShutdown == 1)
  {
    if  (ledErrorBlinkState != 3)
    {
      LED_ErrorStartBlink(3000, 1000); // 3 sekunden
      ledErrorBlinkState = 3;
    }
  }
  else if (sysData.s.cellVoltage > sysData.s.ovpAlarmComp) 
  {
    if  (ledErrorBlinkState != 4)
    {
      LED_ErrorStartBlink(1000, 1000); // 1 sekunden
      ledErrorBlinkState = 4;
    }
  }
  else if (sysData.s.cellVoltage < sysData.s.lvpAlarmComp) 
  {
    if  (ledErrorBlinkState != 5)
    {
      LED_ErrorStartBlink(150, 1000); // 0,15 sekunden
    }
    OUTPUT_LVP_Enable();
    sysData.s.lvpState = 2;
    sysData.s.lvpDelayCounter= sysData.s.lvpTime;
    ledErrorBlinkState = 5;
  }
  else
  {
    LED_ErrorOff();
    ledErrorBlinkState = 0;
  }



  bool epromDataChanged = false;
  //Neu: Implementierung Cycle counter
  //Wenn Zelle leer war und jetzt wieder aufgeladen, dann Cycle Counter erhhen
  if ((batteryState == 0) && (sysData.s.cellVoltage >= sysData.s.ovpStopComp))
  {
    batteryState = 1;
    sysData.s.cycleCounter++;
    printf("Batterie charged! Cyclecounter++!\n");
    epromDataChanged =  true;
  }

  // Batterie State auf entladen schalten, wenn Zelle leer
  if ((batteryState == 1) && (sysData.s.cellVoltage <= sysData.s.lvpStopComp))
  {
    printf("Batterie discharged!\n");
    batteryState = 0;
  }


  if ((HAL_GetTick() >= lastMinuteTime + MINUTE_TIME_INTERVALL) || initRefresh)
  {
    
    initRefresh = 0;
    lastMinuteTime = HAL_GetTick();
    printf("Minute Intervall checks!\n");

    // Speicher Max und min werte
    if (sysData.s.cellVoltage < sysData.s.minVoltage) 
    {
  
      sysData.s.minVoltage = sysData.s.cellVoltage;
      epromDataChanged =  true;
      printf("New Min Voltage %d Saved\n", sysData.s.cellVoltage);
    }

    if (sysData.s.cellVoltage > sysData.s.maxVoltage) 
    {
  
      sysData.s.maxVoltage = sysData.s.cellVoltage;
      epromDataChanged =  true;
      printf("New Max Voltage %d Saved\n", sysData.s.cellVoltage);
    }

    if (sysData.s.cellTemperature < sysData.s.minTemperature) 
    {
      sysData.s.minTemperature = sysData.s.cellTemperature;
      epromDataChanged =  true;
      printf("New Min Temp. %d Saved\n", sysData.s.cellTemperature);
    }

    if (sysData.s.cellTemperature > sysData.s.maxTemperature) 
    {
      sysData.s.maxTemperature = sysData.s.cellTemperature;
      epromDataChanged =  true;
      printf("New Max Temp. %d Saved\n", sysData.s.cellTemperature);
    }

    // Aufzeichnung kritischer Betriebsbedingungen
    // Temperatur 5C ber Abschalttemperatur (Default 50C + 5C = 55C)
    if ((sysData.s.cellTemperature > sysData.s.otShutdownStart + 50) &&  (sysData.s.criticalOverTempTime < 65535 ))
    {
        sysData.s.criticalOverTempTime++;
        epromDataChanged =  true;
        printf("New Max Temp. %d Saved\n", sysData.s.cellTemperature);
    }

    // Temperatur 5C Unter LastAbschalttemperatur (Default -15C + -5C = -20C)
    if ((sysData.s.cellTemperature < sysData.s.utpDischargeStart - 50)  &&  (sysData.s.criticalUnderTempTime < 65535 ))
    {
        sysData.s.criticalUnderTempTime++;
        epromDataChanged =  true;      
    }

    // Spannung unter lvp alarm 
    if ((sysData.s.cellVoltage < sysData.s.lvpAlarm - 100 )  &&  (sysData.s.criticalUnderVoltageTime < 65535 ))
    {
        sysData.s.criticalUnderVoltageTime++;
        epromDataChanged =  true;      
    }

    // Spannung ber ovp alarm 
    if ((sysData.s.cellVoltage > sysData.s.ovpAlarm + 100 )  &&  (sysData.s.criticalOverVoltageTime < 65535 ))
    {
        sysData.s.criticalOverVoltageTime++;
        epromDataChanged =  true;      
    }

  }



  //Speicher Log Daten maximal jede Stunde, aber auch nur dann, wenn siche Werte, seit dem letztem mal gendert haben
  if (HAL_GetTick() >= lastHourTime + HOUR_TIME_INTERVALL)
  {   
    lastHourTime = HAL_GetTick();
    printf("Hour Intervall checks!\n");
    
    if (epromDataChanged )
    {
      EEPROM_storeLogData();
      epromDataChanged = false;
    }
   
  }
}

//========== End of File =======================================================