/**
  ******************************************************************************
  * @file    sysdata.h
  * @author  ECS - Zed Kazharov
  * @version V1.0.0
  * @date    10-Feb-2023
  * @brief   System Daten
  *
  *          Beschreibung          
  *          enthlt alle Systemdaten. Diese sind via modbus auslesbar und vernderbar
  *
  ******************************************************************************
  */ 
#ifndef __SYSDATA_H
#define __SYSDATA_H


//--- INCLUDEs -----------------------------------------------------------------
#include "main.h"
#include "modbus.h"

//--- GLOBALE TYPE DEFS --------------------------------------------------------   

// !!! Tabellenstruktur von LiPro1-x V2 als Vorlage !!!
typedef  struct __attribute__ ((__packed__))
{ 	
  //---- GERTE INFORMATIONEN ---
  uint16_t	      	deviceTypeId;               //0
  uint16_t	      	reserved0;                  //1     Reserviert Alignement
  uint32_t		deviceSn;                   //2-3   Seriennummer
  uint16_t              fwMajor;                    //4     FW Versionsnummer
  uint16_t              fwMinor;                    //5     FW Versionsnummer
  uint16_t              fwRevision;                 //6     FW Versionsnummer
  uint16_t              reserved1;                  //7     RESERVIERT

  //--- GERTE PARAMETER ALLGEMEIN ---
  uint32_t              baudrate;                   //8-9   Baudraute
  uint16_t              parityMode;                 //10    Parity Mode 101 = 'e' = even, 111 = 'o' = odd, 110 = 'n' = none, default even
  uint16_t              stopBits;                   //11    Aktuell nicht ausgewert, automatische Auswahl nach Modbus standard abhngig vom Parity Mode
  uint16_t              slaveAddress;               //12    Slave Adresse
  uint16_t              command;                    //13    Commnd
  uint16_t              reserved2[6];               //14-19 RESERVIERT

  //--- GERTE PARAMETER GERTE SPEZIFISCH ---
  uint16_t              ovpAlarm;                   //20        Gespeicherter Parameter: OVP Alarm Spannung in mV
  uint16_t              lvpAlarm;                   //21        Gespeicherter Parameter: LVP Alarm Spannung in mV 
  uint16_t              ovpStart;                   //22        Gespeicherter Parameter: OVP Start Spannung in mV
  uint16_t              ovpStop;                    //23        Gespeicherter Parameter: OVP Stopp Spannung in mV
  uint16_t              lvpStart;                   //24        Gespeicherter Parameter: LVP Start Spannung in mV
  uint16_t              lvpStop;                    //25        Gespeicherter Parameter: LVP Stopp Spannung in mV
  uint16_t              vShutdownStart;             //26        Gespeicherter Parameter: Unterspannungsabschaltung Start in mV
  uint16_t              vShutdownStop;              //27        Gespeicherter Parameter: Unterspannungsabschaltung Stopp in mV
  int16_t               otShutdownStart;            //28        Gespeicherter Parameter: bertemperaturabschaltung Start in C * 100
  int16_t               otShutdownStop;             //29        Gespeicherter Parameter: bertemperaturabschaltung Stopp in C * 100
  int16_t               utpChargeStart;             //30        Gespeicherter Parameter: Untertemperaturabschaltung Ladung Start in mV -- AB HIER NEUE PARAMETER ---
  int16_t               utpChargeStop;              //31        Gespeicherter Parameter: Untertemperaturabschaltung Ladung Stopp in mV
  int16_t               utpDischargeStart;          //32        Gespeicherter Parameter: Untertemperaturabschaltung Entladung Start in mV
  int16_t               utpDischargeStop;           //33        Gespeicherter Parameter: Untertemperaturabschaltung Entladung Stopp in mV
  uint16_t              lvpTime;                    //34        Gespeicherter Parameter: LVP Verzgerung in s
  uint16_t              defaultBalVoltage;          //35        Gespeicherter Parameter: Ausgleichsspannung Defaultwert in mV. Wird verwendet wenn kein Mastergert vorhanden ist
  int16_t               refTemp;                    //36        Gespeicherter Parameter: Referenz Temperatur
  int16_t               ovpAlarmTempco;             //37        Gespeicherter Parameter: Temperaturkompensation OVP Alarmspannung in mV / C
  int16_t               lvpAlarmTempco;             //38        Gespeicherter Parameter: Temperaturkompensation LVP ALarmspannung in mV / C
  int16_t               ovpStartTempco;             //39        Gespeicherter Parameter: Temperaturkompensation OVP Startspannung in mV / C
  int16_t               ovpStopTempco;              //40        Gespeicherter Parameter: Temperaturkompensation OVP Stoppspannung in mV / C
  int16_t               lvpStartTempco;             //41        Gespeicherter Parameter: Temperaturkompensation LVP Startspannung in mV / C
  int16_t               lvpStopTempco;              //42        Gespeicherter Parameter: Temperaturkompensation LVP Stoppspannung in mV / C
  int16_t               vShutdownStartTempco;       //43        Gespeicherter Parameter: Temperaturkompensation Unterspannungsabschaltung Start in  mV / C
  int16_t               vShutdownStopTempco;        //44        Gespeicherter Parameter: Temperaturkompensation Unterspannungsabschaltung Stopp in  mV / C
  int16_t               balancerVoltageTempco;      //45        Gespeicherter Parameter: Temperaturkompensation Ausgleichsspannung (Standalone Mode) in mV / C
  uint16_t              balancerVoltageRemote;      //46        Remote Setpoint Ausgleichsspannung, diese Register dient zur Steuerung der Balancerspannung von Mastergert
  uint16_t              writeLocked;                //47        1= (Einige) Register sind geschtzt, 0 = Kein Schutz vorhanden
  uint16_t              keyAccepted;                //48        1= lockKey is correct, gesperrte Register sind freigegeben zum beschreiben
  uint16_t              lockKey;                    //49        EINGABE ENTSPERR CODE
  uint16_t              newLockKey;                 //50        EINGABE NEUER Sperrcode
  uint16_t              reserved3[49];              //51-99 RESERVIERT

  // --- AKTUELLE DATEN ab Adresse 100 ---
  uint16_t		cellVoltage;                //100       Zellspannung in mV
  int16_t		cellTemperature;            //101       Zelltemperatur
  int16_t               balancerCurrent;            //102       Aktueller Ausgleichsstrom in mA --> Neuer Istwert
  uint16_t              otShutdown;                 //103       bertemperatur Shutdown Status, 0 = AUS, 1 = AKTIV
  uint16_t              utChargeShutdown;           //104       Untertemperatur Ladung Shutdown Status, 0 = AUS, 1 = AKTIV
  uint16_t              utDischargeShutdown;        //105       Untertemperatur Entladung Shutdown Status, 0 = AUS, 1 = AKTIV
  uint16_t              voltageShutdown;            //106       Spannuns shutdown Status, 0 = AUS, 1 = AKTIV
  uint16_t              lvpDelayCounter;            //107       Zhler bis Abschaltung LVP in Sekunden, ACHTUNG, EINHEIT GENDERT IN SEKUNDEN
  uint16_t              balancerPower;              //108       Nicht wirklich Power in Watt, sondern DAC Value von 0 bis MAX_DAC_VALUE, Achtung Bereich gendert
  uint16_t              mode;                       //109       Modus 0 = Autmatisch. Modus 1 = Manuelle Steuerung ber Register 12 mglich
  uint16_t              lvpState;                   //110       Zustand Schaltausgang LVP, 0 = LVP aus, 1 = LVP aktiv (verzgert), 2 = LVP aktiv (unverzgert)
  uint16_t              ovpState;                   //111       Zustand Schaltausgang OVP, 0 = OVP aus, 1 = LVP aktiv
  uint16_t              maxVoltage;                 //112       Hchste Spannung seit letztem Reset
  uint16_t              minVoltage;                 //113       Niedrigste Spannung seit letztem Reset 
  int16_t               maxTemperature;             //114       Hchste Temperatur seit letztem Reset
  int16_t               minTemperature;             //115       Niedrigste Temperatur seit letztem Reset

  // --- AKTUELLE BERECHNETE PARAMTER ---
  uint16_t              balancerVoltage;            //116       Aktuelle Ausgleichsspannung
  uint16_t              ovpAlarmComp;               //117       OVP Alarmspannung Temperatur kompensiert
  uint16_t              lvpAlarmComp;               //118       LVP Alarmspannung Temperatur kompensiert
  uint16_t              ovpStartComp;               //119       OVP Startspannung Temperatur kompensiert
  uint16_t              ovpStopComp;                //120       OVP Stoppspannung Temperatur kompensiert
  uint16_t              lvpStartComp;               //121       LVP Startspannung Temperatur kompensiert
  uint16_t              lvpStopComp;                //122       LVP Stoppspannung Temperatur kompensiert
  uint16_t              vShutdownStartComp;         //123       Spannungsabschaltung Start Temperatur kompensiert
  uint16_t              vShutdownStopComp;          //124       Spannungsabschaltung Stopp Temperatur kompensiert
  uint16_t              balancerVoltageComp;        //125       Ausgleichsspannung Temperatur kompensiert (wenn im Standalone Mode)  
  uint16_t              masterMode;                 //126       Master modus dedektiert
  uint16_t              converterError;             //127       Fehler DC / DC Converter

  // --- weitere historische Daten
  uint16_t              cycleCounter;               //128
  uint16_t              criticalUnderTempTime;      //129
  uint16_t              criticalOverTempTime;       //130
  uint16_t              criticalUnderVoltageTime;   //131
  uint16_t              criticalOverVoltageTime;    //132
  uint16_t              reserved6[17];              //133-149   RESERVIERT fr weitere wichtige Daten

  // --- AKTUELLE BERECHNETE PARAMTER, Weniger Wichtig, fr DEBUG Zwecke, normalerweise keine bertragung notwendig ---
  uint16_t              chargeTime;                 //150       Interner Debbug Wert: Ladezeit DC DC Converter
  uint16_t              chargeTransferTime;         //151       Interner Debbug Wert: Ladungstransfer Zeit DC DC Converter
  uint16_t              sekVoltage;                 //152
  uint16_t              SekHvErrorCounter;          //153
  uint16_t              SekLvErrorCounter;          //154
  uint16_t              cellVoltageUnfiltered;      //155
}sys_data_struct_t;

typedef union sys_data_union
{
        sys_data_struct_t	s; 
        bword_t mb[sizeof( sys_data_struct_t) /2];
} sys_data_t;


//--- GLOBALE DEFINES ----------------------------------------------------------
#define			CMD_NOCMD				0
#define			CMD_SAVE_PARAMETER_TO_EEPROM		1
#define			CMD_APPLY_MB_PARAMETER			2
#define			CMD_RESET_FACTORY			3
#define                 CMD_RESET                               4
#define                 CMD_SAVE_LOCK_KEY                       5
#define                 CMD_RESET_COUNTERS                      6
#define                 CMD_RESET_MIN_MAX_VALUES                7
#define			CMD_SAVE_SN				12345




    
#define MB_BAUDRATE_DEFAULT					19200	// bit pro sekunde
#define MB_PARITY_MODE_DEFAULT					'e'	// EVEN  - GERADE
#define MB_STOPBIT_MODE_DEFAULT                                 1
#define MB_SLAVE_ADDRESS_DEFAULT				1
#define MB_MASTER_MODE_DEFAULT					0
#define MIN_VOLTAGE_DEFAULT					65535
#define MAX_VOLTAGE_DEFAULT					0
#define MIN_TEMPERATURE_DEFAULT					32767
#define MAX_TEMPERATURE_DEFAULT					-32767



// ------ DEFAULT CELLPARAMETER  ------
#define OVP_ALARM_VOLTAGE_DEFAULT				3700    // [mV] 
#define LVP_ALARM_VOLTAGE_DEFAULT				2800	// [mV] 
#define OVP_START_VOLTAGE_DEFAULT				3650    // [mV] 
#define OVP_STOPP_VOLTAGE_DEFAULT				3600	// [mV] 
#define LVP_START_VOLTAGE_DEFAULT				2900	// [mV] 
#define LVP_STOPP_VOLTAGE_DEFAULT				3200    // [mV] 
#define BAL_VOLTAGE_DEFAULT					3600    // [mV] fr 14,4 V Ladeschlussspannung
#define VOLTAGE_SHUTDOWN_START_VOLTAGE_DEFAULT			2500    // [mV] 
#define VOLTAGE_SHUTDOWN_STOPP_VOLTAGE_DEFAULT			2600    // [mV] 
#define OVERTEMPPROTECTION_START_DEFAULT			500     // [C *10]
#define OVERTEMPPROTECTION_STOP_DEFAULT                         450     // [C *10]
#define UNDERTEMPPROTECTION_CHARGE_START_DEFAULT		50      // [C *10]
#define UNDERTEMPPROTECTION_CHARGE_STOP_DEFAULT			70      // [C *10]
#define UNDERTEMPPROTECTION_DISCHARGE_START_DEFAULT		-150    // [C *10]
#define UNDERTEMPPROTECTION_DISCHARGE_STOP_DEFAULT		-100    // [C *10]
#define LVP_TIME_DEFAULT					600     // s
#define BASIS_TEMP_DEFAULT                                      250     // 25C
#define OVP_ALARM_TEMPCO_DEFAULT                                0       // [mV / C]
#define OVP_START_TEMPCO_DEFAULT                                0       // [mV / C]
#define OVP_STOPP_TEMPCO_DEFAULT                                0       // [mV / C]
#define LVP_ALARM_TEMPCO_DEFAULT                                0       // [mV / C]
#define LVP_START_TEMPCO_DEFAULT                                0       // [mV / C]
#define LVP_STOPP_TEMPCO_DEFAULT                                0       // [mV / C]
#define V_SHUTDOWN_START_TEMPCO_DEFAULT                         0       // [mV / C]
#define V_SHUTDOWN_STOPP_TEMPCO_DEFAULT                         0       // [mV / C]
#define BAL_TEMPCO_DEFAULT                                      0       // [mV / C]


#define CONVERTER_ERROR_NONE                                    0
#define CONVERTER_ERROR_OUTPUT_VOLT_TOO_HIGH                    2
#define CONVERTER_ERROR_OUTPUT_VOLT_TOO_LOW                     3
#define CONVERTER_ERROR_STARTUP_ERROR                           4


//--- GLOBALE VARIABLEN PROTOTYPEN ---------------------------------------------
extern sys_data_t sysData;


#define	MIN_ADRESS                                              0
#define	MAX_ADRESS                                              ((sizeof(sysData)/2)-1)

extern accessMode_t accessModeTable[MAX_ADRESS+1];

    
//--- GLOBALE FUNKTIONS PROTOTYPEN ---------------------------------------------
void SYSDATA_Init(void);




#endif