#include <cstdio>
#include <cstdint>

#include "main.h"
#include "FreeRTOS.h"
#include "task.h"

#include "eth_task.h"

#include "loopback.h"
#include "wizchip_conf.h"
#include "dhcp.h"


static constexpr unsigned ETH_MAX_BUF_SIZE = 2048U;
static constexpr unsigned SOCKET_DHCP = 1;

static constexpr uint16_t delay_ms = 500U;


static uint8_t dhcp_buffer[ETH_MAX_BUF_SIZE];

wiz_NetInfo gWIZNETINFO = {
		.mac = {0x00, 0x08, 0xdc, 0x6f, 0x00, 0x8a},
		.ip = {192, 168, 11, 109},
		.sn = {255, 255, 255, 0},
		.gw = {192, 168, 11, 1},
		.dns = {8, 8, 8, 8},
		.dhcp = NETINFO_DHCP
};

static void ip_assigned(void);
static void ip_updated(void);
static void ip_conflict(void);
static void wizchip_initialize(void);
static void wizchip_reset(void);
static void wizchip_check(void);
static void wizchip_dhcp_init(void);

//------------------------------------------------------------------------------

void ethTaskStart(void* argument)
{
	UNUSED(argument);

	wizchip_initialize();

	if (gWIZNETINFO.dhcp == NETINFO_DHCP) // DHCP
	{
		wizchip_dhcp_init();

		while (DHCP_run() != DHCP_IP_LEASED)
	      wiz_delay_ms(1000);
	}
	else // static
	{
		ctlnetwork(CN_SET_NETINFO, &gWIZNETINFO);
		printf("\r\n----------STATIC Net Information--------------\r\n");
		//print_network_information();
	}

	while (1)
	{
		if (gWIZNETINFO.dhcp == NETINFO_DHCP) DHCP_run();

		loopback_tcps(2, dhcp_buffer, 5000U);

		vTaskDelay(delay_ms);
	}
}

//------------------------------------------------------------------------------

void wizchip_reset(void)
{
	HAL_GPIO_WritePin(ETH_SPI_PWR_GPIO_Port, ETH_SPI_PWR_Pin, GPIO_PIN_RESET);
	vTaskDelay(pdMS_TO_TICKS(100U));
	HAL_GPIO_WritePin(ETH_SPI_RST_GPIO_Port, ETH_SPI_RST_Pin, GPIO_PIN_SET);
	vTaskDelay(pdMS_TO_TICKS(65U));	  // Min 60.3ms
}

//------------------------------------------------------------------------------

void wizchip_initialize(void)
{
	uint8_t W5100S_AdrSet[2][4]= {{2,2,2,2},{2,2,2,2}};
    uint8_t tmp1, tmp2;
	intr_kind temp= IK_DEST_UNREACH;

	wizchip_reset();

	if (ctlwizchip(CW_INIT_WIZCHIP, (void*)W5100S_AdrSet) == -1)
		printf(">>>>W5100s memory initialization failed\r\n");

	if(ctlwizchip(CW_SET_INTRMASK,&temp) == -1)
		printf("W5100S interrupt\r\n");

	wizchip_check();

	while(1)
	{
		ctlwizchip(CW_GET_PHYLINK, &tmp1 );
		ctlwizchip(CW_GET_PHYLINK, &tmp2 );
		if (tmp1 == PHY_LINK_ON && tmp2 == PHY_LINK_ON) break;
	}
}

//------------------------------------------------------------------------------

static void ip_assigned(void)
{
	printf("IP-address was assigned.\n");

	uint8_t ip[4];
	getIPfromDHCP(ip);
	printf("IP-address: %u.%u.%u.%u\n", ip[0], ip[1], ip[2], ip[3]);

	uint8_t nm[4];
	getSNfromDHCP(nm);
	printf("Subnet mask: %u.%u.%u.%u\n", nm[0], nm[1], nm[2], nm[3]);

	uint8_t gw[4];
	getGWfromDHCP(gw);
	printf("Gateway address: %u.%u.%u.%u\n", gw[0], gw[1], gw[2], gw[3]);

	uint8_t dns[4];
	getDNSfromDHCP(dns);
	printf("DNS address: %u.%u.%u.%u\n", dns[0], dns[1], dns[2], dns[3]);

	printf("Lease time: %u\n", getDHCPLeasetime());
}

//------------------------------------------------------------------------------

static void ip_updated(void)
{
	printf("IP-address was updated.\n");
}

//------------------------------------------------------------------------------

static void ip_conflict(void)
{
	printf("IP-address conflict!.\n");
}

//------------------------------------------------------------------------------

void wizchip_check(void)
{
	// Read version register
    if (getVER() != 0x51) // W5100S
    {
        printf(" ACCESS ERR : VERSIONR != 0x51, read value = 0x%02x\n", getVER());
        while (1);
    }
}

//------------------------------------------------------------------------------

void wizchip_dhcp_init(void)
{
    DHCP_init(SOCKET_DHCP, dhcp_buffer);
    reg_dhcp_cbfunc(ip_assigned, ip_updated, ip_conflict);
}

//------------------------------------------------------------------------------
